const express = require('express');
const { authenticateToken, requireStaffOrAdmin } = require('../middleware/auth');
const { mysqlHelper } = require('../database/mysql-helper');
const path = require('path');
const { validationResult, body, param } = require('express-validator');

const router = express.Router();
const DB_PATH = process.env.DB_PATH || './database/usasya.db';

// Validation rules
const idValidation = [
  param('id').isInt({ min: 1 }).withMessage('Valid category ID is required')
];

const createCategoryValidation = [
  body('name').trim().isLength({ min: 2, max: 50 }).withMessage('Category name must be between 2 and 50 characters'),
  body('description').optional().trim().isLength({ max: 500 }).withMessage('Description must be less than 500 characters'),
  body('is_active').optional().isBoolean().withMessage('is_active must be a boolean value')
];

const updateCategoryValidation = [
  body('name').optional().trim().isLength({ min: 2, max: 50 }).withMessage('Category name must be between 2 and 50 characters'),
  body('description').optional().trim().isLength({ max: 500 }).withMessage('Description must be less than 500 characters'),
  body('is_active').optional().isBoolean().withMessage('is_active must be a boolean value')
];

// Get all categories
router.get('/', authenticateToken, async (req, res) => {
  try {
    
    const query = `
      SELECT 
        c.*,
        COUNT(p.id) as product_count
      FROM categories c
      LEFT JOIN products p ON c.id = p.category_id AND p.status = 'active'
      GROUP BY c.id
      ORDER BY c.name
    `;
    
    const rows = await new Promise((resolve, reject) => {
      mysqlHelper.all(query, [], (err, rows) => {
        if (err) reject(err);
        else resolve(rows);
      });
    });

    res.json({ categories: rows });

  } catch (error) {
    console.error('Error fetching categories:', error);
    res.status(500).json({ error: 'Failed to fetch categories' });
  }
});

// Get all categories (public endpoint for clients)
router.get('/public', async (req, res) => {
  try {
    
    const query = `
      SELECT 
        c.id,
        c.name,
        c.description,
        COUNT(p.id) as product_count
      FROM categories c
      LEFT JOIN products p ON c.id = p.category_id AND p.status = 'active'
      WHERE c.is_active = 1
      GROUP BY c.id
      HAVING product_count > 0
      ORDER BY c.name
    `;
    
    const rows = await new Promise((resolve, reject) => {
      mysqlHelper.all(query, [], (err, rows) => {
        if (err) reject(err);
        else resolve(rows);
      });
    });

    res.json({ categories: rows });

  } catch (error) {
    console.error('Error fetching public categories:', error);
    res.status(500).json({ error: 'Failed to fetch categories' });
  }
});

// Get category by ID
router.get('/:id', authenticateToken, async (req, res) => {
  try {
    const categoryId = req.params.id;
    
    const row = await new Promise((resolve, reject) => {
      mysqlHelper.get(
        'SELECT * FROM categories WHERE id = ?',
        [categoryId],
        (err, row) => {
          if (err) reject(err);
          else resolve(row);
        }
      );
    });


    if (!row) {
      return res.status(404).json({ error: 'Category not found' });
    }
    
    res.json({ category: row });

  } catch (error) {
    console.error('Error fetching category:', error);
    res.status(500).json({ error: 'Failed to fetch category' });
  }
});

// Create new category (admin only)
router.post('/', authenticateToken, requireStaffOrAdmin, createCategoryValidation, async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }

    const { name, description, is_active = true } = req.body;

    // Convert undefined to null for MySQL compatibility
    const sanitizedDescription = description === undefined ? null : description;
    const sanitizedIsActive = is_active === undefined ? true : is_active;

    // Check if category name already exists
    const existingCategory = await new Promise((resolve, reject) => {
      mysqlHelper.get('SELECT id FROM categories WHERE name = ?', [name], (err, row) => {
        if (err) reject(err);
        else resolve(row);
      });
    });

    if (existingCategory) {
      return res.status(409).json({ error: 'Category name already exists' });
    }

    // Insert new category
    const result = await new Promise((resolve, reject) => {
      mysqlHelper.run(
        'INSERT INTO categories (name, description, is_active, created_at, updated_at) VALUES (?, ?, ?, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)',
        [name, sanitizedDescription, sanitizedIsActive],
        function(err, result) {
          if (err) reject(err);
          else resolve({ id: result.lastID });
        }
      );
    });

    // Get created category
    const newCategory = await new Promise((resolve, reject) => {
      mysqlHelper.get(
        'SELECT id, name, description, is_active, created_at, updated_at FROM categories WHERE id = ?',
        [result.id],
        (err, row) => {
          if (err) reject(err);
          else resolve(row);
        }
      );
    });


    res.status(201).json({
      message: 'Category created successfully',
      category: newCategory
    });

  } catch (error) {
    console.error('Error creating category:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

// Update category (admin only)
router.put('/:id', authenticateToken, requireStaffOrAdmin, [...idValidation, ...updateCategoryValidation], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }

    const { id } = req.params;
    const { name, description, is_active } = req.body;

    // Check if category exists
    const existingCategory = await new Promise((resolve, reject) => {
      mysqlHelper.get('SELECT * FROM categories WHERE id = ?', [id], (err, row) => {
        if (err) reject(err);
        else resolve(row);
      });
    });

    if (!existingCategory) {
      return res.status(404).json({ error: 'Category not found' });
    }

    // Check name uniqueness if updating name
    if (name && name !== existingCategory.name) {
      const nameExists = await new Promise((resolve, reject) => {
        mysqlHelper.get('SELECT id FROM categories WHERE name = ? AND id != ?', [name, id], (err, row) => {
          if (err) reject(err);
          else resolve(row);
        });
      });

      if (nameExists) {
        return res.status(409).json({ error: 'Category name already exists' });
      }
    }

    // Prepare update fields
    const updateFields = [];
    const updateValues = [];

    if (name !== undefined) {
      updateFields.push('name = ?');
      updateValues.push(name);
    }

    if (description !== undefined) {
      updateFields.push('description = ?');
      updateValues.push(description === null ? null : description);
    }

    if (is_active !== undefined) {
      updateFields.push('is_active = ?');
      updateValues.push(is_active);
    }

    // Always update the updated_at timestamp
    updateFields.push('updated_at = CURRENT_TIMESTAMP');

    if (updateFields.length === 1) { // Only updated_at
      return res.status(400).json({ error: 'No fields to update' });
    }

    // Build and execute update query
    const updateQuery = `UPDATE categories SET ${updateFields.join(', ')} WHERE id = ?`;
    updateValues.push(id);

    await new Promise((resolve, reject) => {
      mysqlHelper.run(updateQuery, updateValues, function(err, result) {
        if (err) reject(err);
        else resolve({ changes: result.changes });
      });
    });

    // Get updated category
    const updatedCategory = await new Promise((resolve, reject) => {
      mysqlHelper.get(
        'SELECT id, name, description, is_active, created_at, updated_at FROM categories WHERE id = ?',
        [id],
        (err, row) => {
          if (err) reject(err);
          else resolve(row);
        }
      );
    });


    res.json({
      message: 'Category updated successfully',
      category: updatedCategory
    });

  } catch (error) {
    console.error('Error updating category:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

// Delete category (admin only)
router.delete('/:id', authenticateToken, requireStaffOrAdmin, idValidation, async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }

    const { id } = req.params;

    // Check if category exists
    const existingCategory = await new Promise((resolve, reject) => {
      mysqlHelper.get('SELECT id FROM categories WHERE id = ?', [id], (err, row) => {
        if (err) reject(err);
        else resolve(row);
      });
    });

    if (!existingCategory) {
      return res.status(404).json({ error: 'Category not found' });
    }

    // Check if category has products
    const productsCount = await new Promise((resolve, reject) => {
      mysqlHelper.get('SELECT COUNT(*) as count FROM products WHERE category_id = ?', [id], (err, row) => {
        if (err) reject(err);
        else resolve(row.count);
      });
    });

    if (productsCount > 0) {
      return res.status(400).json({ 
        error: 'Cannot delete category with existing products',
        productsCount: productsCount
      });
    }

    // Delete category
    await new Promise((resolve, reject) => {
      mysqlHelper.run('DELETE FROM categories WHERE id = ?', [id], function(err, result) {
        if (err) reject(err);
        else resolve({ changes: result.changes });
      });
    });


    res.json({ message: 'Category deleted successfully' });

  } catch (error) {
    console.error('Error deleting category:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

module.exports = router;
